/*
 * Decompiled with CFR 0.152.
 */
package net.perspective.draw.svg;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.SwingUtilities;
import net.perspective.draw.CanvasView;
import net.perspective.draw.DrawingCanvas;
import net.perspective.draw.MainFrame;
import net.perspective.draw.geom.Edge;
import net.perspective.draw.geom.FigureType;
import net.perspective.draw.geom.Grouped;
import net.perspective.draw.svg.CompositePathHandler;
import net.perspective.draw.svg.SubPathInfo;
import net.perspective.draw.svg.SubPathIterator;
import org.apache.batik.anim.dom.SAXSVGDocumentFactory;
import org.apache.batik.ext.awt.geom.ExtendedGeneralPath;
import org.apache.batik.parser.AWTTransformProducer;
import org.apache.batik.parser.ParseException;
import org.apache.batik.parser.PathHandler;
import org.apache.batik.parser.PathParser;
import org.apache.batik.util.XMLResourceDescriptor;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.svg.SVGDocument;

public class SVGUtils {
    private final DrawingCanvas canvas;
    private final CanvasView view;
    private double shifted;
    private static final Logger logger = Logger.getLogger(SVGUtils.class.getName());

    public SVGUtils(MainFrame application) {
        this.canvas = application.getCanvas();
        this.view = this.canvas.getView();
        this.shifted = 20.0;
    }

    public void parseSVG(File file) throws IOException {
        String parser = XMLResourceDescriptor.getXMLParserClassName();
        SAXSVGDocumentFactory factory = new SAXSVGDocumentFactory(parser);
        SVGDocument document = factory.createSVGDocument(file.toURI().toString());
        this.parseSVG(document, false);
    }

    public void parseSVG(String filename, double shifted) {
        this.shifted = shifted;
        try (InputStream in = this.getClass().getResourceAsStream("/svg/" + filename);){
            String parser = XMLResourceDescriptor.getXMLParserClassName();
            SAXSVGDocumentFactory factory = new SAXSVGDocumentFactory(parser);
            SVGDocument document = factory.createSVGDocument(null, in);
            this.parseSVG(document, true);
        }
        catch (IOException e) {
            logger.log(Level.WARNING, "Error reading SVG file: {0}", filename);
            e.printStackTrace();
        }
    }

    protected void parseSVG(SVGDocument document, boolean fill) throws IOException {
        Element root = document.getDocumentElement();
        NodeList paths = root.getElementsByTagName("path");
        Grouped group = new Grouped();
        for (int i = 0; i < paths.getLength(); ++i) {
            Element path = (Element)paths.item(i);
            List<SubPathInfo> subPathInfos = this.handlePathWithInfo(path);
            int j = 0;
            while (j < subPathInfos.size()) {
                SubPathInfo subPathInfo = subPathInfos.get(j);
                Edge figure = new Edge(FigureType.VECTOR);
                AffineTransform at = subPathInfo.getTransform();
                Shape transformed = at.createTransformedShape(subPathInfo.getPath2D());
                figure.setPath(new Path2D.Double(transformed));
                if (!fill) {
                    figure.setColor(subPathInfo.getStrokeColor());
                } else {
                    figure.setColor(this.canvas.getFillColor());
                }
                figure.setStroke(subPathInfo.getStroke());
                if (!fill && subPathInfo.getFillColor() != null) {
                    figure.setFillColor(subPathInfo.getFillColor());
                    figure.setTransparency(100);
                } else if (!fill && subPathInfo.getFillColor() == null) {
                    figure.setFillColor(new Color(1.0f, 1.0f, 1.0f, 0.0f));
                    figure.setTransparency(0);
                } else if (fill) {
                    figure.setFillColor(this.canvas.getFillColor());
                    figure.setTransparency(100);
                }
                figure.setEndPoints();
                group.addDrawItem(figure);
                int pathIndex = i;
                int subPathIndex = j++;
                logger.log(Level.FINEST, "Added figure from SVG path {0} subpath {1}: {2}", new Object[]{pathIndex, subPathIndex, figure});
            }
        }
        if (fill) {
            double height = group.getEnd().y - group.getStart().y;
            double scale = 64.0 / height;
            group.setScale(scale);
        }
        group.moveTo(this.shifted, this.shifted);
        this.shifted += 20.0;
        SwingUtilities.invokeLater(() -> {
            this.view.setNewItem(group);
            this.view.copyItemToCanvas(group);
        });
    }

    private List<SubPathInfo> handlePathWithInfo(Element path) {
        logger.finest("Starting to handle SVG path element with subpath info");
        String d = path.getAttribute("d");
        String fillRule = path.getAttribute("fill-rule");
        List<Path2D.Double> paths = this.handlePath(path);
        ArrayList<SubPathInfo> subPathInfos = new ArrayList<SubPathInfo>();
        if (this.shouldTreatAsCompositePath(d, fillRule)) {
            for (int i = 0; i < paths.size(); ++i) {
                Path2D.Double path2D = paths.get(i);
                if (path2D == null) continue;
                subPathInfos.add(new SubPathInfo(path2D, i, path, false, this));
                logger.log(Level.FINEST, "Created SubPathInfo for composite path {0}", i);
            }
        } else {
            int i;
            PathAnalysis analysis = this.analyzeSubPathsForHoles(paths);
            for (i = 0; i < analysis.outerPaths.size(); ++i) {
                subPathInfos.add(new SubPathInfo(analysis.outerPaths.get(i), i, path, false, this));
                logger.log(Level.FINEST, "Created SubPathInfo for outer path {0}", i);
            }
            for (i = 0; i < analysis.holePaths.size(); ++i) {
                subPathInfos.add(new SubPathInfo(analysis.holePaths.get(i), analysis.outerPaths.size() + i, path, true, this));
                logger.log(Level.FINEST, "Created SubPathInfo for hole {0}", i);
            }
        }
        logger.log(Level.FINEST, "Created {0} SubPathInfo objects", subPathInfos.size());
        return subPathInfos;
    }

    private List<Path2D.Double> handlePath(Element path) {
        logger.finest("Starting to handle SVG path element");
        String d = path.getAttribute("d");
        String fillRule = path.getAttribute("fill-rule");
        if (fillRule == null || fillRule.isEmpty()) {
            fillRule = "nonzero";
        }
        if (d == null || d.isEmpty()) {
            logger.warning("No path data found in SVG element.");
            return new ArrayList<Path2D.Double>();
        }
        if (this.shouldTreatAsCompositePath(d, fillRule)) {
            return this.handleCompositePathWithHoles(d, fillRule);
        }
        return this.handleSeparateSubPaths(d);
    }

    private boolean shouldTreatAsCompositePath(String pathData, String fillRule) {
        boolean hasClosedSubPaths;
        long moveToCount = pathData.chars().filter(ch -> ch == 77 || ch == 109).count();
        boolean hasEvenOddRule = "evenodd".equals(fillRule);
        boolean bl = hasClosedSubPaths = pathData.contains("Z") || pathData.contains("z");
        if (moveToCount > 1L) {
            if (hasEvenOddRule || hasClosedSubPaths) {
                return true;
            }
            if (pathData.matches(".*[Aa].*") && moveToCount >= 2L) {
                return true;
            }
            if (this.hasImplicitClosedPaths(pathData)) {
                return true;
            }
            if (this.hasPotentiallyContainedSubPaths(pathData)) {
                return true;
            }
        }
        return false;
    }

    private boolean hasImplicitClosedPaths(String pathData) {
        String[] subPaths;
        for (String subPath : subPaths = pathData.split("(?=[Mm])")) {
            if (subPath.trim().isEmpty() || !this.isImplicitlyClosed(subPath.trim())) continue;
            return true;
        }
        return false;
    }

    private boolean isImplicitlyClosed(String subPath) {
        try {
            long segmentCount = subPath.chars().filter(ch -> "LlCcQqAaHhVvSsTt".indexOf(ch) >= 0).count();
            boolean hasMultipleSegments = segmentCount >= 2L;
            boolean isCircularPath = subPath.contains("a") || subPath.contains("A");
            boolean likelyReturnToStart = hasMultipleSegments && (isCircularPath || segmentCount >= 4L);
            return likelyReturnToStart;
        }
        catch (IllegalArgumentException | NullPointerException e) {
            logger.log(Level.FINE, "Error analyzing subpath closure: {0}", e.getMessage());
            return false;
        }
    }

    private boolean hasPotentiallyContainedSubPaths(String pathData) {
        try {
            List<Rectangle2D> bounds = this.getSubPathBounds(pathData);
            if (bounds.size() < 2) {
                return false;
            }
            for (int i = 0; i < bounds.size(); ++i) {
                for (int j = 0; j < bounds.size(); ++j) {
                    if (i == j || !bounds.get(i).contains(bounds.get(j))) continue;
                    logger.finest("Found potentially contained subpaths - treating as composite");
                    return true;
                }
            }
        }
        catch (IllegalArgumentException | NullPointerException e) {
            logger.log(Level.FINE, "Error analyzing subpath containment: {0}", e.getMessage());
        }
        return false;
    }

    private List<Rectangle2D> getSubPathBounds(String pathData) {
        ArrayList<Rectangle2D> bounds = new ArrayList<Rectangle2D>();
        try {
            String[] subPaths;
            for (String subPath : subPaths = pathData.split("(?=[Mm])")) {
                Rectangle2D subBounds;
                if (subPath.trim().isEmpty() || (subBounds = this.estimateSubPathBounds(subPath.trim())) == null) continue;
                bounds.add(subBounds);
            }
        }
        catch (IllegalArgumentException | NullPointerException e) {
            logger.log(Level.FINE, "Error calculating subpath bounds: {0}", e.getMessage());
        }
        return bounds;
    }

    private Rectangle2D estimateSubPathBounds(String subPath) {
        try {
            Pattern pattern = Pattern.compile("-?\\d+(?:\\.\\d+)?");
            Matcher matcher = pattern.matcher(subPath);
            ArrayList<Double> coords = new ArrayList<Double>();
            while (matcher.find()) {
                coords.add(Double.valueOf(matcher.group()));
            }
            if (coords.size() < 4) {
                return null;
            }
            double minX = (Double)coords.get(0);
            double maxX = (Double)coords.get(0);
            double minY = (Double)coords.get(1);
            double maxY = (Double)coords.get(1);
            for (int i = 0; i < coords.size() - 1; i += 2) {
                double x = (Double)coords.get(i);
                double y = (Double)coords.get(i + 1);
                minX = Math.min(minX, x);
                maxX = Math.max(maxX, x);
                minY = Math.min(minY, y);
                maxY = Math.max(maxY, y);
            }
            return new Rectangle2D.Double(minX, minY, maxX - minX, maxY - minY);
        }
        catch (NumberFormatException e) {
            logger.log(Level.FINE, "Error estimating bounds for subpath: {0}", e.getMessage());
            return null;
        }
    }

    private List<Path2D.Double> handleCompositePathWithHoles(String d, String fillRule) {
        logger.finest("Processing composite path with potential holes");
        Path2D.Double compositePath = new Path2D.Double();
        if ("evenodd".equals(fillRule)) {
            compositePath.setWindingRule(0);
            logger.finest("Set winding rule to EVEN_ODD as specified");
        } else if (this.shouldForceEvenOdd(d)) {
            compositePath.setWindingRule(0);
            logger.finest("Forced winding rule to EVEN_ODD for better hole rendering");
        } else {
            compositePath.setWindingRule(1);
            logger.finest("Set winding rule to NON_ZERO");
        }
        PathParser parser = new PathParser();
        parser.setPathHandler((PathHandler)new CompositePathHandler(compositePath, this));
        try {
            parser.parse(d);
            logger.log(Level.FINEST, "Successfully parsed composite path with {0} winding rule", compositePath.getWindingRule() == 0 ? "EVEN_ODD" : "NON_ZERO");
        }
        catch (ParseException e) {
            logger.log(Level.WARNING, "Error parsing composite path data: {0}", e.getMessage());
        }
        return Collections.singletonList(compositePath);
    }

    private boolean shouldForceEvenOdd(String pathData) {
        long moveToCount = pathData.chars().filter(ch -> ch == 77 || ch == 109).count();
        boolean hasArcs = pathData.toLowerCase().contains("a");
        return moveToCount > 1L && hasArcs;
    }

    private List<Path2D.Double> handleSeparateSubPaths(String d) {
        logger.finest("Processing separate subpaths");
        SubPathIterator subPathIterator = new SubPathIterator(this);
        PathParser parser = new PathParser();
        parser.setPathHandler((PathHandler)subPathIterator);
        try {
            parser.parse(d);
        }
        catch (ParseException e) {
            logger.log(Level.WARNING, "Error parsing separate subpaths data: {0}", e.getMessage());
            return new ArrayList<Path2D.Double>();
        }
        return subPathIterator.getSubPaths();
    }

    protected void addArcToPath(Path2D.Double path, float rx, float ry, float xAxisRotation, boolean largeArcFlag, boolean sweepFlag, float x, float y, boolean isAbsolute) {
        Point2D currentPoint = path.getCurrentPoint();
        if (currentPoint == null) {
            return;
        }
        ExtendedGeneralPath tempPath = new ExtendedGeneralPath();
        tempPath.moveTo((float)currentPoint.getX(), (float)currentPoint.getY());
        tempPath.arcTo(rx, ry, (float)Math.toRadians(xAxisRotation), largeArcFlag, sweepFlag, x, y);
        path.append((Shape)tempPath, true);
    }

    private PathAnalysis analyzeSubPathsForHoles(List<Path2D.Double> subPaths) {
        if (subPaths.size() <= 1) {
            return new PathAnalysis(new ArrayList<Path2D.Double>(subPaths), new ArrayList<Path2D.Double>());
        }
        ArrayList<PathAreaInfo> pathInfos = new ArrayList<PathAreaInfo>();
        for (int i = 0; i < subPaths.size(); ++i) {
            Path2D.Double path = subPaths.get(i);
            double area = this.calculatePathArea(path);
            pathInfos.add(new PathAreaInfo(path, i, area));
        }
        pathInfos.sort((a, b) -> Double.compare(b.area, a.area));
        ArrayList<Path2D.Double> outerPaths = new ArrayList<Path2D.Double>();
        ArrayList<Path2D.Double> holePaths = new ArrayList<Path2D.Double>();
        for (int i = 0; i < pathInfos.size(); ++i) {
            PathAreaInfo info = (PathAreaInfo)pathInfos.get(i);
            if (i == 0 || info.area > ((PathAreaInfo)pathInfos.get((int)0)).area * 0.1) {
                outerPaths.add(info.path);
                continue;
            }
            holePaths.add(info.path);
        }
        return new PathAnalysis(outerPaths, holePaths);
    }

    private double calculatePathArea(Path2D path) {
        Rectangle2D bounds = path.getBounds2D();
        return bounds.getWidth() * bounds.getHeight();
    }

    private float extractStrokeWidth(Element element) {
        String styleAttr;
        logger.finest("Extracting stroke width");
        float strokeWidth = 1.0f;
        String strokeAttr = element.getAttribute("stroke");
        if (strokeAttr != null && (strokeAttr.equals("none") || strokeAttr.equals("transparent"))) {
            logger.log(Level.FINEST, "Stroke set to none or transparent, setting stroke width to 0");
            return 0.0f;
        }
        String strokeWidthAttr = element.getAttribute("stroke-width");
        if (strokeWidthAttr != null && !strokeWidthAttr.isEmpty()) {
            try {
                strokeWidth = Float.parseFloat(strokeWidthAttr);
                logger.log(Level.FINEST, "Found stroke width attribute on element: {0}", Float.valueOf(strokeWidth));
                return strokeWidth;
            }
            catch (NumberFormatException e) {
                logger.log(Level.WARNING, "Invalid stroke-width value: {0}", strokeWidthAttr);
            }
        }
        if ((styleAttr = element.getAttribute("style")) != null && !styleAttr.isEmpty()) {
            logger.log(Level.FINEST, "Checking style attribute for stroke-width: {0}", styleAttr);
            float width = this.extractStrokeWidthFromStyle(styleAttr);
            if (width != -1.0f) {
                logger.log(Level.FINEST, "Found stroke width in style: {0}", Float.valueOf(strokeWidth));
                return width;
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                String parentStrokeAttr = parent.getAttribute("stroke");
                if (parentStrokeAttr != null && (parentStrokeAttr.equals("none") || parentStrokeAttr.equals("transparent"))) {
                    logger.log(Level.FINEST, "Parent stroke set to none or transparent, setting stroke width to 0");
                    return 0.0f;
                }
                strokeWidthAttr = parent.getAttribute("stroke-width");
                if (strokeWidthAttr != null && !strokeWidthAttr.isEmpty()) {
                    try {
                        strokeWidth = Float.parseFloat(strokeWidthAttr);
                        logger.log(Level.FINEST, "Found stroke width attribute on parent group: {0}", Float.valueOf(strokeWidth));
                        return strokeWidth;
                    }
                    catch (NumberFormatException e) {
                        logger.log(Level.WARNING, "Invalid stroke-width value in parent: {0}", strokeWidthAttr);
                    }
                }
                if ((styleAttr = parent.getAttribute("style")) != null && !styleAttr.isEmpty()) {
                    logger.log(Level.FINEST, "Checking parent style attribute for stroke-width: {0}", styleAttr);
                    float width = this.extractStrokeWidthFromStyle(styleAttr);
                    if (width != -1.0f) {
                        logger.log(Level.FINEST, "Found stroke width in style: {0}", Float.valueOf(strokeWidth));
                        return width;
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        logger.log(Level.FINEST, "Using default stroke width: {0}", Float.valueOf(strokeWidth));
        return strokeWidth;
    }

    private float extractStrokeWidthFromStyle(String styleAttr) {
        String[] styles;
        for (String style : styles = styleAttr.split(";")) {
            String[] keyValue = style.trim().split(":");
            if (keyValue.length != 2) continue;
            String key = keyValue[0].trim();
            String value = keyValue[1].trim();
            if (key.equals("stroke") && (value.equals("none") || value.equals("transparent"))) {
                logger.log(Level.FINEST, "Style stroke set to none or transparent, setting stroke width to 0");
                return 0.0f;
            }
            if (!key.equals("stroke-width")) continue;
            try {
                return Float.parseFloat(value);
            }
            catch (NumberFormatException e) {
                logger.log(Level.WARNING, "Invalid style stroke-width value: {0}", value);
            }
        }
        return -1.0f;
    }

    protected Color extractStrokeColor(Element element) {
        String styleAttr;
        logger.finest("Extracting stroke color");
        Color strokeColor = Color.BLACK;
        String strokeAttr = element.getAttribute("stroke");
        if (strokeAttr != null && !strokeAttr.isEmpty()) {
            try {
                strokeColor = this.parseColor(strokeAttr);
                logger.log(Level.FINEST, "Found stroke color attribute on element: {0}", strokeColor);
                return strokeColor;
            }
            catch (IllegalArgumentException | NullPointerException e) {
                logger.log(Level.WARNING, "Invalid stroke color value: {0}", strokeAttr);
            }
        }
        if ((styleAttr = element.getAttribute("style")) != null && !styleAttr.isEmpty()) {
            logger.log(Level.FINEST, "Checking style attribute for stroke color: {0}", styleAttr);
            Color color = this.extractStrokeColorFromStyle(styleAttr);
            if (color != null) {
                logger.log(Level.FINEST, "Found stroke color in style: {0}", color);
                return color;
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                strokeAttr = parent.getAttribute("stroke");
                if (strokeAttr != null && !strokeAttr.isEmpty()) {
                    try {
                        strokeColor = this.parseColor(strokeAttr);
                        logger.log(Level.FINEST, "Found stroke color attribute on parent group: {0}", strokeColor);
                        return strokeColor;
                    }
                    catch (IllegalArgumentException | NullPointerException e) {
                        logger.log(Level.WARNING, "Invalid stroke color value in parent: {0}", strokeAttr);
                    }
                }
                if ((styleAttr = parent.getAttribute("style")) != null && !styleAttr.isEmpty()) {
                    logger.log(Level.FINEST, "Checking parent style attribute for stroke color: {0}", styleAttr);
                    Color color = this.extractStrokeColorFromStyle(styleAttr);
                    if (color != null) {
                        logger.log(Level.FINEST, "Found stroke color in parent style: {0}", color);
                        return color;
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        logger.log(Level.FINEST, "Using default stroke color: {0}", strokeColor);
        return strokeColor;
    }

    private Color extractStrokeColorFromStyle(String styleAttr) {
        String[] styles;
        for (String style : styles = styleAttr.split(";")) {
            String[] keyValue = style.trim().split(":");
            if (keyValue.length != 2) continue;
            String key = keyValue[0].trim();
            String value = keyValue[1].trim();
            if (!key.equals("stroke")) continue;
            try {
                return this.parseColor(value);
            }
            catch (IllegalArgumentException | NullPointerException e) {
                logger.log(Level.WARNING, "Invalid style stroke color value: {0}", value);
            }
        }
        return null;
    }

    protected Optional<Color> extractFillColor(Element element) {
        String styleAttr;
        String fillAttr;
        logger.finest("Extracting fill color");
        String fillOpacityAttr = element.getAttribute("fill-opacity");
        if (fillOpacityAttr != null && !fillOpacityAttr.isEmpty()) {
            try {
                float opacity = Float.parseFloat(fillOpacityAttr);
                if (opacity <= 0.0f) {
                    logger.log(Level.FINEST, "Found fill-opacity=0 on element, returning empty Optional");
                    return Optional.empty();
                }
            }
            catch (NumberFormatException e) {
                logger.log(Level.WARNING, "Invalid fill-opacity value: {0}", fillOpacityAttr);
            }
        }
        if ((fillAttr = element.getAttribute("fill")) != null && !fillAttr.isEmpty()) {
            if (fillAttr.equalsIgnoreCase("none")) {
                logger.log(Level.FINEST, "Found fill='none' on element");
                return Optional.empty();
            }
            if (fillAttr.equalsIgnoreCase("transparent")) {
                logger.log(Level.FINEST, "Found fill='transparent' on element, returning empty Optional");
                return Optional.empty();
            }
            try {
                Color fillColor = this.parseColor(fillAttr);
                logger.log(Level.FINEST, "Found fill color attribute on element: {0}", fillColor);
                return Optional.of(fillColor);
            }
            catch (IllegalArgumentException | NullPointerException e) {
                logger.log(Level.WARNING, "Invalid fill color value: {0}", fillAttr);
            }
        }
        if ((styleAttr = element.getAttribute("style")) != null && !styleAttr.isEmpty()) {
            logger.log(Level.FINEST, "Checking style attribute for fill color: {0}", styleAttr);
            if (styleAttr.matches(".*fill-opacity\\s*:\\s*0(?:\\.0*)?\\s*[;$].*")) {
                logger.log(Level.FINEST, "Found fill-opacity:0 in style, returning empty Optional");
                return Optional.empty();
            }
            Optional<Color> colorOpt = this.extractFillColorFromStyle(styleAttr);
            if (colorOpt.isPresent()) {
                logger.log(Level.FINEST, "Found fill color in style: {0}", colorOpt.get());
                return colorOpt;
            }
            if (styleAttr.contains("fill:transparent") || styleAttr.contains("fill: transparent")) {
                logger.log(Level.FINEST, "Found fill:transparent in style, returning empty Optional");
                return Optional.empty();
            }
            if (styleAttr.contains("fill:none") || styleAttr.contains("fill: none")) {
                logger.log(Level.FINEST, "Found fill:none in style");
                return Optional.empty();
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                fillOpacityAttr = parent.getAttribute("fill-opacity");
                if (fillOpacityAttr != null && !fillOpacityAttr.isEmpty()) {
                    try {
                        float opacity = Float.parseFloat(fillOpacityAttr);
                        if (opacity <= 0.0f) {
                            logger.log(Level.FINEST, "Found fill-opacity=0 on parent, returning empty Optional");
                            return Optional.empty();
                        }
                    }
                    catch (NumberFormatException e) {
                        logger.log(Level.WARNING, "Invalid fill-opacity value in parent: {0}", fillOpacityAttr);
                    }
                }
                if ((fillAttr = parent.getAttribute("fill")) != null && !fillAttr.isEmpty()) {
                    if (fillAttr.equalsIgnoreCase("none")) {
                        logger.log(Level.FINEST, "Found fill='none' on parent");
                        return Optional.empty();
                    }
                    if (fillAttr.equalsIgnoreCase("transparent")) {
                        logger.log(Level.FINEST, "Found fill='transparent' on parent, returning empty Optional");
                        return Optional.empty();
                    }
                    try {
                        Color fillColor = this.parseColor(fillAttr);
                        logger.log(Level.FINEST, "Found fill color attribute on parent group: {0}", fillColor);
                        return Optional.of(fillColor);
                    }
                    catch (IllegalArgumentException | NullPointerException e) {
                        logger.log(Level.WARNING, "Invalid fill color value in parent: {0}", fillAttr);
                    }
                }
                if ((styleAttr = parent.getAttribute("style")) != null && !styleAttr.isEmpty()) {
                    logger.log(Level.FINEST, "Checking parent style attribute for fill color: {0}", styleAttr);
                    if (styleAttr.matches(".*fill-opacity\\s*:\\s*0(?:\\.0*)?\\s*[;$].*")) {
                        logger.log(Level.FINEST, "Found fill-opacity:0 in parent style, returning empty Optional");
                        return Optional.empty();
                    }
                    Optional<Color> colorOpt = this.extractFillColorFromStyle(styleAttr);
                    if (colorOpt.isPresent()) {
                        logger.log(Level.FINEST, "Found fill color in parent style: {0}", colorOpt.get());
                        return colorOpt;
                    }
                    if (styleAttr.contains("fill:transparent") || styleAttr.contains("fill: transparent")) {
                        logger.log(Level.FINEST, "Found fill:transparent in parent style, returning empty Optional");
                        return Optional.empty();
                    }
                    if (styleAttr.contains("fill:none") || styleAttr.contains("fill: none")) {
                        logger.log(Level.FINEST, "Found fill:none in parent style");
                        return Optional.empty();
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        logger.log(Level.FINEST, "No fill color found, returning empty Optional");
        return Optional.empty();
    }

    private Optional<Color> extractFillColorFromStyle(String styleAttr) {
        String[] styles = styleAttr.split(";");
        boolean hasFillColor = false;
        Color fillColor = null;
        for (String style : styles) {
            String[] keyValue = style.trim().split(":");
            if (keyValue.length != 2) continue;
            String key = keyValue[0].trim();
            String value = keyValue[1].trim();
            if (key.equals("fill-opacity")) {
                try {
                    float opacity = Float.parseFloat(value);
                    if (opacity <= 0.0f) {
                        logger.log(Level.FINEST, "Found fill-opacity:0 in style");
                        return Optional.empty();
                    }
                }
                catch (NumberFormatException e) {
                    logger.log(Level.WARNING, "Invalid fill-opacity value in style: {0}", value);
                }
            }
            if (!key.equals("fill")) continue;
            hasFillColor = true;
            if (value.equalsIgnoreCase("none")) {
                logger.log(Level.FINEST, "Found fill:none in style");
                return Optional.empty();
            }
            if (value.equalsIgnoreCase("transparent")) {
                logger.log(Level.FINEST, "Found fill:transparent in style");
                return Optional.empty();
            }
            try {
                fillColor = this.parseColor(value);
            }
            catch (IllegalArgumentException | NullPointerException e) {
                logger.log(Level.WARNING, "Invalid style fill color value: {0}", value);
                hasFillColor = false;
            }
        }
        if (hasFillColor && fillColor != null) {
            return Optional.of(fillColor);
        }
        return Optional.empty();
    }

    private Color parseColor(String colorStr) {
        logger.log(Level.FINEST, "Parsing color: {0}", colorStr);
        if (colorStr == null || colorStr.equals("none") || colorStr.equals("transparent")) {
            return Color.BLACK;
        }
        if (colorStr.startsWith("#")) {
            return this.parseHexColor(colorStr);
        }
        if (colorStr.startsWith("rgb(")) {
            return this.parseRgbColor(colorStr);
        }
        return this.parseNamedColor(colorStr);
    }

    private Color parseHexColor(String hexColor) {
        String hex = hexColor.substring(1);
        if (hex.length() == 3) {
            char r = hex.charAt(0);
            char g = hex.charAt(1);
            char b = hex.charAt(2);
            hex = new String(new char[]{r, r, g, g, b, b});
        }
        try {
            return new Color(Integer.parseInt(hex.substring(0, 2), 16), Integer.parseInt(hex.substring(2, 4), 16), Integer.parseInt(hex.substring(4, 6), 16));
        }
        catch (IllegalArgumentException | StringIndexOutOfBoundsException e) {
            logger.log(Level.WARNING, "Failed to parse hex color: {0}", hexColor);
            return Color.BLACK;
        }
    }

    private Color parseRgbColor(String rgbColor) {
        try {
            String values = rgbColor.substring(4, rgbColor.length() - 1);
            String[] components = values.split(",");
            if (components.length >= 3) {
                int r = this.parseRgbComponent(components[0]);
                int g = this.parseRgbComponent(components[1]);
                int b = this.parseRgbComponent(components[2]);
                return new Color(r, g, b);
            }
        }
        catch (IllegalArgumentException | StringIndexOutOfBoundsException e) {
            logger.log(Level.WARNING, "Failed to parse RGB color: {0}", rgbColor);
        }
        return Color.BLACK;
    }

    private int parseRgbComponent(String component) {
        if ((component = component.trim()).endsWith("%")) {
            float percent = Float.parseFloat(component.substring(0, component.length() - 1)) / 100.0f;
            return Math.round(percent * 255.0f);
        }
        return Integer.parseInt(component);
    }

    private Color parseNamedColor(String colorName) {
        switch (colorName.toLowerCase()) {
            case "black": {
                return Color.BLACK;
            }
            case "silver": {
                return new Color(192, 192, 192);
            }
            case "gray": {
                return Color.GRAY;
            }
            case "white": {
                return Color.WHITE;
            }
            case "maroon": {
                return new Color(128, 0, 0);
            }
            case "red": {
                return Color.RED;
            }
            case "purple": {
                return new Color(128, 0, 128);
            }
            case "fuchsia": {
                return Color.MAGENTA;
            }
            case "green": {
                return new Color(0, 128, 0);
            }
            case "lime": {
                return Color.GREEN;
            }
            case "olive": {
                return new Color(128, 128, 0);
            }
            case "yellow": {
                return Color.YELLOW;
            }
            case "navy": {
                return new Color(0, 0, 128);
            }
            case "blue": {
                return Color.BLUE;
            }
            case "teal": {
                return new Color(0, 128, 128);
            }
            case "aqua": {
                return Color.CYAN;
            }
        }
        logger.log(Level.WARNING, "Unknown color name: {0}, defaulting to black", colorName);
        return Color.BLACK;
    }

    private Element getParentElement(Element element) {
        if (element.getParentNode() == null || !(element.getParentNode() instanceof Element)) {
            return null;
        }
        return (Element)element.getParentNode();
    }

    private int extractStrokeLineJoin(Element element) {
        logger.finest("Extracting stroke line join");
        int lineJoin = 0;
        String lineJoinAttr = element.getAttribute("stroke-linejoin");
        if (lineJoinAttr != null && !lineJoinAttr.isEmpty()) {
            lineJoin = this.parseLineJoin(lineJoinAttr);
            logger.log(Level.FINEST, "Found line join attribute on element: {0}", lineJoinAttr);
            return lineJoin;
        }
        String styleAttr = element.getAttribute("style");
        if (styleAttr != null && !styleAttr.isEmpty()) {
            logger.log(Level.FINEST, "Checking style attribute for line join: {0}", styleAttr);
            int join = this.extractLineJoinFromStyle(styleAttr);
            if (join >= 0) {
                logger.log(Level.FINEST, "Found line join in style: {0}", join);
                return join;
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                lineJoinAttr = parent.getAttribute("stroke-linejoin");
                if (lineJoinAttr != null && !lineJoinAttr.isEmpty()) {
                    lineJoin = this.parseLineJoin(lineJoinAttr);
                    logger.log(Level.FINEST, "Found line join attribute on parent group: {0}", lineJoinAttr);
                    return lineJoin;
                }
                styleAttr = parent.getAttribute("style");
                if (styleAttr != null && !styleAttr.isEmpty()) {
                    logger.log(Level.FINEST, "Checking parent style attribute for line join: {0}", styleAttr);
                    int join = this.extractLineJoinFromStyle(styleAttr);
                    if (join >= 0) {
                        logger.log(Level.FINEST, "Found line join in parent style: {0}", join);
                        return join;
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        logger.log(Level.FINEST, "Using default line join: {0}", lineJoin);
        return lineJoin;
    }

    private int extractLineJoinFromStyle(String styleAttr) {
        String[] styles;
        for (String style : styles = styleAttr.split(";")) {
            String[] keyValue = style.trim().split(":");
            if (keyValue.length != 2) continue;
            String key = keyValue[0].trim();
            String value = keyValue[1].trim();
            if (!key.equals("stroke-linejoin")) continue;
            return this.parseLineJoin(value);
        }
        return -1;
    }

    private int parseLineJoin(String lineJoin) {
        if (lineJoin == null) {
            return 0;
        }
        switch (lineJoin.toLowerCase()) {
            case "miter": {
                return 0;
            }
            case "round": {
                return 1;
            }
            case "bevel": {
                return 2;
            }
        }
        logger.log(Level.WARNING, "Unknown line join: {0}, defaulting to miter", lineJoin);
        return 0;
    }

    private int extractStrokeLineCap(Element element) {
        logger.finest("Extracting stroke line cap");
        int lineCap = 0;
        String lineCapAttr = element.getAttribute("stroke-linecap");
        if (lineCapAttr != null && !lineCapAttr.isEmpty()) {
            lineCap = this.parseLineCap(lineCapAttr);
            logger.log(Level.FINEST, "Found line cap attribute on element: {0}", lineCapAttr);
            return lineCap;
        }
        String styleAttr = element.getAttribute("style");
        if (styleAttr != null && !styleAttr.isEmpty()) {
            logger.log(Level.FINEST, "Checking style attribute for line cap: {0}", styleAttr);
            int cap = this.extractLineCapFromStyle(styleAttr);
            if (cap >= 0) {
                logger.log(Level.FINEST, "Found line cap in style: {0}", cap);
                return cap;
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                lineCapAttr = parent.getAttribute("stroke-linecap");
                if (lineCapAttr != null && !lineCapAttr.isEmpty()) {
                    lineCap = this.parseLineCap(lineCapAttr);
                    logger.log(Level.FINEST, "Found line cap attribute on parent group: {0}", lineCapAttr);
                    return lineCap;
                }
                styleAttr = parent.getAttribute("style");
                if (styleAttr != null && !styleAttr.isEmpty()) {
                    logger.log(Level.FINEST, "Checking parent style attribute for line cap: {0}", styleAttr);
                    int cap = this.extractLineCapFromStyle(styleAttr);
                    if (cap >= 0) {
                        logger.log(Level.FINEST, "Found line cap in parent style: {0}", cap);
                        return cap;
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        logger.log(Level.FINEST, "Using default line cap: {0}", lineCap);
        return lineCap;
    }

    private int extractLineCapFromStyle(String styleAttr) {
        String[] styles;
        for (String style : styles = styleAttr.split(";")) {
            String[] keyValue = style.trim().split(":");
            if (keyValue.length != 2) continue;
            String key = keyValue[0].trim();
            String value = keyValue[1].trim();
            if (!key.equals("stroke-linecap")) continue;
            return this.parseLineCap(value);
        }
        return -1;
    }

    private int parseLineCap(String lineCap) {
        if (lineCap == null) {
            return 0;
        }
        switch (lineCap.toLowerCase()) {
            case "butt": {
                return 0;
            }
            case "round": {
                return 1;
            }
            case "square": {
                return 2;
            }
        }
        logger.log(Level.WARNING, "Unknown line cap: {0}, defaulting to butt", lineCap);
        return 0;
    }

    protected BasicStroke createStrokeFromElement(Element element) {
        logger.finest("Creating BasicStroke from SVG element");
        float strokeWidth = this.extractStrokeWidth(element);
        int lineJoin = this.extractStrokeLineJoin(element);
        int lineCap = this.extractStrokeLineCap(element);
        float[] dashArray = this.extractStrokeDashArray(element);
        float dashPhase = 0.0f;
        if (dashArray != null) {
            logger.log(Level.FINEST, "Creating dashed stroke with width={0}, cap={1}, join={2}, dash pattern length={3}", new Object[]{Float.valueOf(strokeWidth), lineCap, lineJoin, dashArray.length});
            return new BasicStroke(strokeWidth, lineCap, lineJoin, 10.0f, dashArray, dashPhase);
        }
        logger.log(Level.FINEST, "Creating solid stroke with width={0}, cap={1}, join={2}", new Object[]{Float.valueOf(strokeWidth), lineCap, lineJoin});
        return new BasicStroke(strokeWidth, lineCap, lineJoin);
    }

    private float[] extractStrokeDashArray(Element element) {
        String dashArrayAttr = element.getAttribute("stroke-dasharray");
        if (dashArrayAttr != null && !dashArrayAttr.isEmpty() && !dashArrayAttr.equals("none")) {
            return this.parseDashArray(dashArrayAttr);
        }
        String styleAttr = element.getAttribute("style");
        if (styleAttr != null && !styleAttr.isEmpty()) {
            String[] styles;
            for (String style : styles = styleAttr.split(";")) {
                String[] keyValue = style.trim().split(":");
                if (keyValue.length != 2) continue;
                String key = keyValue[0].trim();
                String value = keyValue[1].trim();
                if (!key.equals("stroke-dasharray") || value.equals("none")) continue;
                return this.parseDashArray(value);
            }
        }
        Element parent = this.getParentElement(element);
        while (parent != null) {
            if (parent.getTagName().equals("g")) {
                dashArrayAttr = parent.getAttribute("stroke-dasharray");
                if (dashArrayAttr != null && !dashArrayAttr.isEmpty() && !dashArrayAttr.equals("none")) {
                    return this.parseDashArray(dashArrayAttr);
                }
                styleAttr = parent.getAttribute("style");
                if (styleAttr != null && !styleAttr.isEmpty()) {
                    String[] styles;
                    for (String style : styles = styleAttr.split(";")) {
                        String[] keyValue = style.trim().split(":");
                        if (keyValue.length != 2) continue;
                        String key = keyValue[0].trim();
                        String value = keyValue[1].trim();
                        if (!key.equals("stroke-dasharray") || value.equals("none")) continue;
                        return this.parseDashArray(value);
                    }
                }
            }
            parent = this.getParentElement(parent);
        }
        return null;
    }

    private float[] parseDashArray(String dashArrayStr) {
        logger.log(Level.FINEST, "Parsing dash array: {0}", dashArrayStr);
        try {
            String[] parts = dashArrayStr.trim().split("[ ,]+");
            float[] result = new float[parts.length];
            for (int i = 0; i < parts.length; ++i) {
                result[i] = Float.parseFloat(parts[i]);
            }
            boolean allZero = true;
            for (float f : result) {
                if (!(f > 0.0f)) continue;
                allZero = false;
                break;
            }
            if (allZero || result.length == 0) {
                return null;
            }
            return result;
        }
        catch (NumberFormatException e) {
            logger.log(Level.WARNING, "Failed to parse dash array: {0}", dashArrayStr);
            return null;
        }
    }

    protected AffineTransform getPathTransform(Element element) {
        AffineTransform at = new AffineTransform();
        Element current = element;
        while (current != null && current.getNodeType() == 1) {
            String transformStr = current.getAttribute("transform");
            if (transformStr != null && !transformStr.isEmpty()) {
                try {
                    AffineTransform local = AWTTransformProducer.createAffineTransform((String)transformStr);
                    at.preConcatenate(local);
                }
                catch (ParseException e) {
                    logger.log(Level.WARNING, "Error parsing path data: {0}", e.getMessage());
                }
            }
            if (current.getParentNode() instanceof Element) {
                current = (Element)current.getParentNode();
                continue;
            }
            current = null;
        }
        return at;
    }

    private static class PathAnalysis {
        final List<Path2D.Double> outerPaths;
        final List<Path2D.Double> holePaths;

        PathAnalysis(List<Path2D.Double> outerPaths, List<Path2D.Double> holePaths) {
            this.outerPaths = outerPaths;
            this.holePaths = holePaths;
        }
    }

    private static class PathAreaInfo {
        final Path2D.Double path;
        final int index;
        final double area;

        PathAreaInfo(Path2D.Double path, int index, double area) {
            this.path = path;
            this.index = index;
            this.area = area;
        }
    }
}

